% Copyright 2016 Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000 with Sandia
% Corporation, the U.S. Government retains certain rights in this software

% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are
% met:
% 
%     (1) Redistributions of source code must retain the above copyright
%     notice, this list of conditions and the following disclaimer. 
% 
%     (2) Redistributions in binary form must reproduce the above copyright
%     notice, this list of conditions and the following disclaimer in
%     the documentation and/or other materials provided with the
%     distribution.  
%     
%     (3)The name of the author may not be used to
%     endorse or promote products derived from this software without
%     specific prior written permission.
% 
% THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
% IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
% WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
% DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
% INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
% (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
% SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
% HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
% STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
% IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
% POSSIBILITY OF SUCH DAMAGE.

function [decontaminatedD,nIters] = despikeHyperUBS(D,maxComponents,readoutSigma,maxPixelFraction)

%FUNCTION [decontaminatedD,nIters] = despikeHyperUBS(D,maxComponents,readoutSigma,maxPixelFraction)
%
% PURPOSE:
%   despikeHyperUBS is designed to take hyperspectral data and remove 
%	cosmic spikes. This function runs the UBS-DM-HS algorithm (or the 
%	UBS-DM algorithm depending upon setting) through multiple iterations 
%	until a final stable result is achieved. Each iteration attenuates 
%	cosmic spikes, but multiple cycles are often needed to reduce cosmic 
%	spikes to the typical noise-level variations. The algorithm assumes 
%	that many spectra (pixels) have been recorded, and that each spectral 
%	component of interest is present in multiple pixels. 
%
% DEPENDENCIES:
%	despikeHyperUBSiteration.m
%	corrColumns.m
%
% CLASSES:
%	-None-
%
% INPUTS:
%   D:
%		An m by n matrix, where each of the n columns corresponds to a 
%		spectrum. The data is expected to be contaminated with cosmic
%		spikes. 
%   maxComponents: 
%       The maximum number of expected spectral components. 
%   readoutSigma:
%       The expected readout noise standard deviation, typically the
%       readout noise of the camera. 
%	maxPixelFraction:
%		Pixels which are responsible for greater than this fraction of the
%		total weight (positive or negative portions) of a spectral
%		component must correspond to spikes. The amount of that spectral
%		component in that pixel is then set to zero. The default value for
%		the UBS-DM-HS algorithm is 10/(the number of spectra). 
%
% OUTPUTS:
%   decontaminatedD:
%       A matrix equivalent to D, but with cosmic spikes attenuated. 
%	nIters:
%		The number of iterations of the algorithm required before a stable
%		result was reached. 
%
% REFERENCES:
%   This code was originally based off the following reference for the
%   UBS-DM algorithm. 
%   1)  Zhang, D. M. and D. Ben-Amotz (2002). "Removal of cosmic spikes 
%       from hyper-spectral images using a hybrid upper-bound spectrum 
%       method." Applied Spectroscopy 56(1): 91-98.
%	This code can be used to run the UBS-DM algorithm (with very minor 
%	modifications) by setting maxPixelFraction=1. However, the code is
%	actually an implementation of the UBS-DM-HS algorithm, which is more
%	robust and generally performs better. The UBS-DM-HS algorithm is
%	documented in the following reference:
%	2)	Anthony, S.M and Timlin, J.A. (2016). "Removing Cosmic Spikes 
%		Using a Hyperspectral Upper-Bound Spectrum Method." Applied
%		Spectroscopy
%
% PROGRAMMER COMMENTS:
%	-None-
%
% LIMITATIONS:
%	1) despikeHyperUBSiteration is designed for hyperspectral data. It 
%		assumes that many spectra (pixels) have been recorded, and that 
%		each spectral component of interest is present in multiple pixels. 
%	2) The elimination of components from pixels that dominate that
%		component can result in reductions below baseline. The data should
%		be baseline corrected as much as possible first. 

% Revision:
%	Sandia Hyper-UBS 1.0.0

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Provide defaults. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%This is generally a reasonable value for the UBS-DM-HS algorithm. 
if ~exist('maxPixelFraction','var')
	maxPixelFraction = 10/size(D,2);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Initialize for the loop. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%Record the original data, which is needed in each iteration, and transpose
%to the needed orientation. 
originalD = D';

%Indicate whether to continue. 
notDone = true;

%Initialize the spectra to use. 
currentD = D;

%Count the number of iterations. 
nIters = 0; 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Loop through, running repeated iterations of the UBS-DM-HS algorithm. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

while notDone
	%Perform an iteration of the algorithm. 
	newD = despikeHyperUBSiteration(currentD',maxComponents,readoutSigma,maxPixelFraction,originalD)';
	
	%Update the count
	nIters = nIters+1;
	
	%Determine whether the algorithm has reached a stable result. 
	if 	mean(corrColumns(newD,currentD))>(1-1e-6)
		%A stable result has been reached. Terminate. 
		notDone = false;
		decontaminatedD = newD;
	else
		%Another iteration is necessary. Set up for it. 
		currentD = newD;
	end
end
